/*
  Analog_Keypad.cpp

  Open Source 2016, Bent Arnoldsen
  UC Holstebro - Tekniks Gymnasium

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <Analog_Keypad.h>


// <<constructor>> Allows custom keymap, pin configuration, and keypad sizes.
Analog_Keypad::Analog_Keypad(char *userKeyMap, byte analogPin, byte numKeys) {
	inputPin = analogPin;
	numOfKeys = numKeys;
	keymap = userKeyMap;
	keyOffset = 512 / numOfKeys;
	segment = 256 / numOfKeys;

	begin();

	setDebounceTime(200);
	setHoldTime(1000);
	setHoldMode(false);
	setNumAccept(5);

	startTime = millis();
}

// <<constructor>> Additional constructor, allowing the number of polls to accept key.
Analog_Keypad::Analog_Keypad(char *userKeyMap, byte analogPin, byte numKeys, byte numAccept) {
	inputPin = analogPin;
	numOfKeys = numKeys;
	keyOffset = 512 / numOfKeys;
	segment = 256 / numOfKeys;

	begin();

	setDebounceTime(300);
	setHoldTime(1000);
	setHoldMode(false);
	setNumAccept(numAccept);

	startTime = millis();
}

// Let the user define a keymap
void Analog_Keypad::begin() {
}

// Returns a single key only. Retained for backwards compatibility.
char Analog_Keypad::getKey() {
	thisKey = getKeyNum();
	if (thisKey == NO_KEY) {
		lastKey = thisKey;
		numPolls = 0;
		return NO_KEY;
	}
	if (thisKey == lastKey) {
		numPolls++;
		if (numPolls == numAccept) {	// Wait for 5 similar readings
			startTime = millis();
			repeating = false;
			return keymap[thisKey];
		} else if (numPolls > numAccept) {
			numPolls = numAccept;
			if (holdMode) {
				if (repeating) {		// Send a key each debounceTime i repeating has started
					if ((millis() - startTime) > debounceTime) {
						startTime += debounceTime;
						return keymap[thisKey];
					} else {
						return NO_KEY;
					}
				} else {				// Start repeating if we are over the holdTime
					if ((millis() - startTime) > holdTime) {
						repeating = true;
						startTime += holdTime;
						return keymap[thisKey];
					} else {
						return NO_KEY;
					}
				}
			} else {
				return NO_KEY;
			}
		}
	} else {  		// New key reading - restart validation
		lastKey = thisKey;
		numPolls = 0;
	}
	return NO_KEY;
}

// For use if the user wants to get a pressed / not pressed functionality
bool Analog_Keypad::isPressed(char keyChar) {
	char key = getKeyNum();
	if (key == NO_KEY) return false;
	if (keymap[key] == keyChar) return true;
	return false;	// Not pressed.
}

// For use if the user just wants to wait for a key to be pressed - locks until the key is pressed
char Analog_Keypad::waitForKey() {
	char waitKey = NO_KEY;
	while( (waitKey = getKey()) == NO_KEY );	// Block everything while waiting for a keypress.
	return waitKey;
}

// Method that allows to change how many times the getKey-method should be called, before the key is valid
void Analog_Keypad::setNumAccept(byte count) {
    numAccept = count;
}

// Set / Reset the mode where keyboard can repeat automatically
void Analog_Keypad::setHoldMode(bool mode) {
    holdMode = mode;
}

// Sets the time before a key should start repeating in hold-mode
void Analog_Keypad::setHoldTime(uint hold) {
    holdTime = hold;
}

// Define the time between repeated keys come
// Minimum debounceTime is 1 mS. Any lower *will* slow down the loop().
void Analog_Keypad::setDebounceTime(uint debounce) {
	debounce<1 ? debounceTime=1 : debounceTime=debounce;
}


// Private method that returns the keynumber on the keypad - WARNING - returning NOKEY if the key is not valid
byte Analog_Keypad::getKeyNum() {
	int analogValue = analogRead(inputPin) - keyOffset;
	byte rest;
	byte key;
	
	if (analogValue < 0) return NO_KEY;
	rest = (analogValue / segment) % 4;
	if (rest == 0 || rest == 3) return NO_KEY;
	return analogValue / (1024 / numOfKeys);
}

/*
|| @changelog
*/
